<?php

/**
 * Project: RBI Report
 * Author: Grant McClure <grant.mcclure@t4bi.com>
 * File: rbi_report.lib.php
 * Version: 1.0
 */

require(RBI_REPORT_DIR . 'libs/Wkhtmltopdf.php');
require(RBI_REPORT_DIR . 'libs/pdftk.php');
require(RBI_REPORT_DIR . 'libs/config.php');
require(RBI_REPORT_DIR . 'libs/sugar_api.php');
include_once(RBI_REPORT_DIR . '../../db_config.php');

/**
 * RBI Report
 *
 */
class RBI_Report
{
    /* mls settings */
    var $config;

    // database object
    var $pdo = null;
    // smarty template object
    var $tpl = null;
    // error messages
    var $error = null;
    /* set up array for numbers that need to be prepended by 'an' instead of 'a' */
    var $vowelNumbers = array(8, 11, 18, 80, 81, 82, 82, 83, 84, 85, 86, 87, 88, 89);

    /**
     * class constructor
     */
    function __construct() {
        global $db_url;

        // instantiate the config object with MLS settings
        $this->config = new Config();

        // instantiate the pdo object
        try {
          $url = parse_url($db_url);
          $this->dbtype = "mysql";
          $this->dbuser = urldecode($url['user']);
          $this->dbpass = isset($url['pass']) ? urldecode($url['pass']) : '';
          $this->dbhost = urldecode($url['host']);
          $this->dbname = basename(urldecode($url['path']));

          $dsn = "{$this->dbtype}:host={$this->dbhost};dbname={$this->dbname}";
          $this->pdo = new PDO($dsn,$this->dbuser,$this->dbpass,
                            array(PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8")  );
        } catch (PDOException $e) {
            echo "Error!: " . $e->getMessage();
            die();
        }

        // instantiate the template object
        $this->tpl = new RBI_Reports_Smarty;

    }

    function displayReportSelector()
    {
        // assign error message
        $this->tpl->assign('error', $this->error);

        $this->tpl->assign('webroot', '.');
 
        foreach ($this->config->locations as $loc) {
            $locArray[$loc] = $this->getLocationValues($loc);
        }
        $this->tpl->assign('locArray', $locArray);

        $facetArray = $this->getFacetData();

        $monthArrayTemp = $this->getReportMonths();
        $monthArray = array();
        foreach ($monthArrayTemp as $row) {
            $row["timestamp"] = mktime(0, 0, 0, substr($row["time_period"], 4, 2), 1, substr($row["time_period"], 0, 4));
            $monthArray[] = $row;
        }

        $quarterArrayTemp = $this->getReportQuarters();
        foreach ($quarterArrayTemp as $row) {
            $row["timestamp"] = mktime(0, 0, 0, 1, 1, substr($row["time_period"], 0, 4));
            $quarter = substr($row["time_period"], 4, 1);
            $quarterend = $quarterendArray[$quarter];
            $row["label"] = "Q" . $quarter . " " . substr($row["time_period"], 0, 4);
            $quarterArray[] = $row;
        }

        $this->tpl->assign('facetArray', $facetArray);
        $this->tpl->assign('monthArray', $monthArray);
        $this->tpl->assign('quarterArray', $quarterArray);
        $this->tpl->display("report_selector.tpl");
    }

    function createCoverPage($agg_type, $loc_type, $loc_id, $f_id, $time_period, $uid = null, $sid = null)
    {
        // assign error message
        $this->tpl->assign('error', $this->error);

        # get MLS-specific configuration
        $this->tpl->assign('mls_name', $this->config->mls_name);
        $this->tpl->assign('mls_logo', $this->config->mls_logo);
        $this->tpl->assign('show_logo_on_cover', $this->config->show_logo_on_cover);
        $this->tpl->assign('show_dom_widget', $this->config->show_dom_widget);

        $this->tpl->assign('agg_type', $agg_type);
        $this->loadTemplatePersonalization($uid, $sid);
        $this->loadTemplateLocation($loc_type, $loc_id);
        $this->loadTemplateHomeTypesArray($f_id);
        $prev_time_period = $this->loadTemplateDates($agg_type, $time_period);
    }

    function createReport($agg_type, $loc_type, $loc_id, $f_id, $time_period, $uid = null, $sid = null)
    {
        // assign error message
        $this->tpl->assign('error', $this->error);
       
        // load personalization
        $this->loadTemplatePersonalization($uid, $sid);

        // get MLS-specific configuration
        $this->tpl->assign('mls_name', $this->config->mls_name);
        $this->tpl->assign('mls_logo', $this->config->mls_logo);
        $this->tpl->assign('show_dom_widget', $this->config->show_dom_widget);
        $this->tpl->assign('chartDataBarLabel', $this->config->chart_bar_label);
        $this->tpl->assign('footnote_active', $this->config->footnote_active);
        $this->tpl->assign('footnote_standard', $this->config->footnote_standard);

        // pod placement
        foreach ($this->config->podmap as $pod=>$sections) {
          foreach ($sections as $section) {
           foreach ($section as $section_label=>$rdindex) {
              if ($this->personalizedData == null  || !in_array($rdindex,$this->personalizedData["disabled_metrics"])) {
                  #echo "assigning $section_label to $pod<br>";
                  $this->tpl->assign($section_label, $pod);
                  break 2;
               }
           }
          }
        }

        // pod preference
        //if ($this->personalizedData!=null) {
        //}

        $this->loadTemplateLocation($loc_type, $loc_id);
        $this->loadTemplateHomeType($f_id);
        $prev_time_period = $this->loadTemplateDates($agg_type, $time_period);

        // load vowel array for summary text handling
        $this->tpl->assign('vowelNumbers', $this->vowelNumbers);

        // Get Generation Date
        $gen_date = $this->getGenerationDate($agg_type, $loc_type, $loc_id, $f_id, $time_period);
        $this->tpl->assign('gen_date', $gen_date);

        // Get Period Data (Monthly or Quarterly)
        $monthlyPrevMonthData = $this->getPeriodData($agg_type, $loc_type, $loc_id, $f_id, $prev_time_period);
        $monthlyFiveYearData = $this->getPeriodFiveYearData($agg_type, $loc_type, $loc_id, $f_id, $time_period);
        $this->tpl->assign('monthlyCurrMonth', $monthlyFiveYearData[0]);
        $this->tpl->assign('monthlyPrevMonth', $monthlyPrevMonthData);
        $this->tpl->assign('monthlyPrevYear', $monthlyFiveYearData[1]);
        $monthlyPrevMonthDiff = $this->getDataDiff($monthlyFiveYearData[0], $monthlyPrevMonthData);
        $monthlyPrevMonthDiffDir = $this->getDataDiffDir($monthlyPrevMonthDiff);
/*
print "<pre>";
print "timeperiod: $time_period<BR>";
print "prevtimeperiod: $prev_time_period<BR>";
var_dump($monthlyFiveYearData[0]);
print "<BR>";
var_dump($monthlyPrevMonthData);
print "<BR>";
var_dump($monthlyPrevMonthDiff);
print "<BR>";
print "</pre>";
exit;
*/
        $this->tpl->assign('monthlyPrevMonthDiff', $monthlyPrevMonthDiff);
        $this->tpl->assign('monthlyPrevMonthDiffDir', $monthlyPrevMonthDiffDir);
        $monthlyPrevYearDiff = $this->getDataDiff($monthlyFiveYearData[0], $monthlyFiveYearData[1]);
        $monthlyPrevYearDiffDir = $this->getDataDiffDir($monthlyPrevYearDiff);
        $this->tpl->assign('monthlyPrevYearDiff', $monthlyPrevYearDiff);
        $this->tpl->assign('monthlyPrevYearDiffDir', $monthlyPrevYearDiffDir);
        $monthlyFiveYearStats = $this->getDataStats($monthlyFiveYearData);
        $this->tpl->assign('monthlyFiveYearStats', $monthlyFiveYearStats);

        // Get YTD Data
        if ($agg_type == "MONTHLY") {
            #$ytdPrevMonthData = $this->getYTDData($agg_type, $loc_type, $loc_id, $f_id, $prev_time_period);
            $ytdTwoYearData = $this->getYTDTwoYearData($agg_type, $loc_type, $loc_id, $f_id, $time_period);
        } else if ($agg_type == "QUARTERLY") {
            # get the YTD data for the end of the Quarter
            $qmap_time_period = substr($time_period, 0, 4) . sprintf("%02d", 3 * (substr($time_period, 4, 1)));
            #$qmap_prev_time_period = substr($prev_time_period,0,4) . sprintf("%02d", 3*(substr($prev_time_period,4,1)));
            #$ytdPrevMonthData = $this->getYTDData($agg_type, $loc_type, $loc_id, $qmap_prev_time_period);
            $ytdTwoYearData = $this->getYTDTwoYearData($agg_type, $loc_type, $loc_id, $f_id, $qmap_time_period);
        }
        #$this->tpl->assign('ytdPrevMonth',  $ytdPrevMonthData);
        $this->tpl->assign('ytdCurrYear', $ytdTwoYearData[0]);
        $this->tpl->assign('ytdPrevYear', $ytdTwoYearData[1]);
        $this->tpl->assign('ytdPrevYearDiff', $this->getDataDiff($ytdTwoYearData[0], $ytdTwoYearData[1]));

        // Get Chart Data
        $chartData = $this->getChartData($agg_type, $loc_type, $loc_id, $f_id, $time_period);
        $chart_med_price_string = "";
        $chart_num_sales_string = "";
        $chart_num_pendings_string = "";
        $chart_x_axis_month = "";
        $chart_x_axis_year = "";
        $chartDataBarMin = null;
        $chartDataBarMax = null;
        $chartDataMedianSoldPriceMin = null;
        $chartDataMedianSoldPriceMax = null;
        $i = 0;
        $month_array = array(1 => "J", 2 => "F", 3 => "M", 4 => "A", 5 => "M", 6 => "J", 7 => "J", 8 => "A", 9 => "S", 10 => "O", 11 => "N", 12 => "D");
        $prev_year = "";
        foreach ($chartData as $chartRow) {
            $year = substr($chartRow["time_period"], 0, 4);
            if ($i > 0) {
                $chart_med_price_string .= ",";
                $chart_num_sales_string .= ",";
                $chart_num_pendings_string .= ",";
                $chart_x_axis_month .= ",";
                #if ($year!=$prev_year) {
                $chart_x_axis_year .= ",";
                #}
            }
            $curClosedSalesVal = $chartRow["num_sold_listings"];
            if ($chartDataBarMin == null || $chartDataBarMin > $curClosedSalesVal) {
                $chartDataBarMin = $curClosedSalesVal;
            }
            if ($chartDataBarMax == null || $chartDataBarMax < $curClosedSalesVal) {
                $chartDataBarMax = $curClosedSalesVal;
            }
            $curMedianSoldPriceVal = intval($chartRow["sld_med_sp"] / 1000);
            if ($chartDataMedianSoldPriceMin == null || $chartDataMedianSoldPriceMin > $curMedianSoldPriceVal) {
                $chartDataMedianSoldPriceMin = $curMedianSoldPriceVal;
            }
            if ($chartDataMedianSoldPriceMax == null || $chartDataMedianSoldPriceMax < $curMedianSoldPriceVal) {
                $chartDataMedianSoldPriceMax = $curMedianSoldPriceVal;
            }
            $chart_med_price_string .= $curMedianSoldPriceVal;
            if ($agg_type == "MONTHLY") {
                #$chart_num_pendings_string .= $curNewPendingsVal;
                $chart_num_sales_string .= $curClosedSalesVal;
                $chart_x_axis_month .= "'" . $month_array[intval(substr($chartRow["time_period"], 4, 2))] . "'";
            } else if ($agg_type == "QUARTERLY") {
                $chart_num_sales_string .= $curClosedSalesVal;
                $chart_x_axis_month .= "'" . substr($chartRow["time_period"], 4, 1) . "'";
            }
            #if ($year!=$prev_year) {
            $chart_x_axis_year .= substr($chartRow["time_period"], 0, 4);
            #}
            $prev_year = $year;
            $i++;
        }
        $this->tpl->assign('chartDataMedPrice', $chart_med_price_string);
        $this->tpl->assign('chartDataBarData', $chart_num_sales_string);
        $this->tpl->assign('chartDataXAxisMonth', $chart_x_axis_month);
        $this->tpl->assign('chartDataXAxisYear', $chart_x_axis_year);

        # calculate min/max on y-axes
        $chartDataClosedSalesPow = strlen($chartDataBarMax - $chartDataBarMin) - 2;
        $chartDataBarMin = (floor(($chartDataBarMin / pow(10, $chartDataClosedSalesPow)) - 1) * pow(10, $chartDataClosedSalesPow));
        $chartDataBarMax = (ceil(($chartDataBarMax / pow(10, $chartDataClosedSalesPow)) + 1) * pow(10, $chartDataClosedSalesPow));
        $this->tpl->assign('chartDataBarMin', $chartDataBarMin);
        $this->tpl->assign('chartDataBarMax', $chartDataBarMax);

        $chartDataMedianSoldPricePow = strlen($chartDataMedianSoldPriceMax - $chartDataMedianSoldPriceMin) - 2;
        $chartDataMedianSoldPriceMin = (floor(($chartDataMedianSoldPriceMin / pow(10, $chartDataMedianSoldPricePow)) - 1) * pow(10, $chartDataMedianSoldPricePow));
        $chartDataMedianSoldPriceMax = (ceil(($chartDataMedianSoldPriceMax / pow(10, $chartDataMedianSoldPricePow)) + 1) * pow(10, $chartDataMedianSoldPricePow));

        $this->tpl->assign('chartDataMedianSoldPriceMin', $chartDataMedianSoldPriceMin);
        $this->tpl->assign('chartDataMedianSoldPriceMax', $chartDataMedianSoldPriceMax);
    }

    private function loadTemplatePersonalization($uid,$sid)
    {
        # get personalization data if required
        if ($sid != null) {
            if ($this->personalizedData == null) {
                $pdata = $this->getUserDataSugar($sid);
                $pdataArray = array();
                if ($pdata!=null && is_array($pdata->entry_list)) {
                    foreach ($pdata->entry_list[0]->name_value_list as $key=>$val) {
                       $pdataArray[$key] = $val->value;
                    }
                    $this->personalizedData = $pdataArray;
                }
            }
            #       echo "<PRE>";
            #       print $this->personalizedData["email1"];
            #       var_dump($this->personalizedData);
            #       echo "</PRE>";
            $this->tpl->assign('personalizedData', $this->personalizedData);
        }
/*
        if ($uid != null) {
            if ($this->personalizedData == null) {
                $this->personalizedData = $this->getUserDataSugar($uid);
            }
            #var_dump($this->personalizedData);
            $this->tpl->assign('personalizedData', $this->personalizedData);
        }
*/
    }

    function loadTemplateDates($agg_type, $time_period)
    {
        $yearForTimestamp = substr($time_period, 0, 4);
        $monthForTimestamp = substr($time_period, 4, 2);
        if (!$monthForTimestamp) {
            $monthForTimestamp = 1; // default to January for annual report
        }

        if ($agg_type == "MONTHLY") {
            $time_period_date = $time_period . "01";
            $currMonthTimestamp = DateTime::createFromFormat("Ymd", $time_period_date);
            $prevMonthTimestamp = DateTime::createFromFormat("Ymd", $time_period_date)->sub(new DateInterval('P1M'));
            $prevYearTimestamp = DateTime::createFromFormat("Ymd", $time_period_date)->sub(new DateInterval('P1Y'));
            $this->tpl->assign('currMonthTimestamp', $currMonthTimestamp);
            $this->tpl->assign('prevMonthTimestamp', $prevMonthTimestamp);
            $this->tpl->assign('prevYearTimestamp', $prevYearTimestamp);
            $prev_time_period = $prevMonthTimestamp->format("Ym");
            $period = $currMonthTimestamp->format('F');
        } else if ($agg_type == "QUARTERLY") {
            $endArray = array("1st", "2nd", "3rd", "4th");
            $quarter = $monthForTimestamp; // should use better name; in this case it's really the quarter not the month
            $this->tpl->assign('quarterLong', $endArray[$quarter - 1] . " Quarter");
            $this->tpl->assign('quarterShort', "Q" . $quarter);

            $prev_year = $yearForTimestamp - 1;
            if ($quarter == "1") {
                $prev_period_year = $prev_year;
                $prev_period_quarter = "4";
            } else {
                $prev_period_year = $yearForTimestamp;
                $prev_period_quarter = $quarter - 1;
            }
            $prev_time_period = $prev_period_year . $prev_period_quarter;

            $this->tpl->assign('prevPeriodLabel', "Q" . $prev_period_quarter . " " . $prev_period_year);
            $this->tpl->assign('prevYearLabel', "Q" . $quarter . " " . $prev_year);
            $this->tpl->assign('prevYear', $prev_year);
            $this->tpl->assign('currYear', $yearForTimestamp);
            $this->tpl->assign('currPeriod', "Q" . $quarter);
            $period = "Q" . $quarter;
        }

        #$this->tpl->assign('gen_date', $gen_date);
        return $prev_time_period;
    }

    function loadTemplateLocation($loc_type, $loc_id)
    {
        $locData = $this->getLocationData($loc_type, $loc_id);
        $locationName = $locData['l_desc'];
        $this->tpl->assign('locationName', $locationName);
        $this->tpl->assign('locationType', $loc_type);
    }

    function loadTemplateHomeTypesArray($f_id)
    {
        $facetArray=array();
        $facetData = $this->getFacetData();
        foreach ($facetData as $facet) {
           $facetArray[$facet["f_id"]] = $facet["report_desc"];
        }
        $homeTypes=array();
        $f_ids = explode(",", $f_id);
        foreach ($f_ids as $f_id_mod) {
          if (array_key_exists($f_id_mod, $facetArray)) {
            array_push($homeTypes,$facetArray[$f_id_mod]);
          }
        }
        $this->tpl->assign('homeTypes', $homeTypes);
    }

    function loadTemplateHomeType($f_id)
    {
        if ($f_id != 1000) {
            $facetData = $this->getFacetData();
			foreach ($facetData as $facet) {
				if($facet["f_id"] == $f_id) {
					$homeType = $facet["report_desc"];
					break;
				}
			}
            $this->tpl->assign('homeType', $homeType);
        }
    }

    function getDataDiff($thisDataArray, $lastDataArray)
    {
        $diffDataArray = array();
        if ((is_array($thisDataArray) && count($thisDataArray) > 0)) {
            foreach ($thisDataArray as $key => $value) {
                if (is_array($value)) {
                    $diffDataArray[$key] = $this->getDataDiff($thisDataArray[$key], $lastDataArray[$key]);
                } else {
                    if ($lastDataArray[$key] > 0) {
                        $diffDataArray[$key] = number_format(100.0 * (($thisDataArray[$key]) - ($lastDataArray[$key])) / ($lastDataArray[$key]), 1);
                    } else {
                        $diffDataArray[$key] = number_format(0.0, 1);
                    }
                }
            }
        }
        return $diffDataArray;
    }

    function getDataDiffDir($dataDiffArray)
    {
        $diffdirDataArray = array();
        if ((is_array($dataDiffArray) && count($dataDiffArray) > 0)) {
            foreach ($dataDiffArray as $key => $value) {
                if ($value > 0) {
                    $diffdirDataArray[$key] = "Up";
                } else if ($value < 0) {
                    $diffdirDataArray[$key] = "Down";
                } else {
                    $diffdirDataArray[$key] = "Sideways";
                }
            }
        }
        return $diffdirDataArray;
    }


    function getDataStats($thisDataArrayRows)
    {
        $statsDataArray = array();
        if ((is_array($thisDataArrayRows) && count($thisDataArrayRows) > 0)) {
            $i = 0;
            foreach ($thisDataArrayRows as $thisDataArray) {
                $i++;
                foreach ($thisDataArray as $key => $value) {
                    if (array_key_exists($key, $statsDataArray)) {
                        $statsDataArray[$key]["tot"] += $thisDataArray[$key];
                        if ($statsDataArray[$key]["min"] > $thisDataArray[$key]) {
                            $statsDataArray[$key]["min"] = $thisDataArray[$key];
                        }
                        if ($statsDataArray[$key]["max"] < $thisDataArray[$key]) {
                            $statsDataArray[$key]["max"] = $thisDataArray[$key];
                        }
                    } else {
                        $statsDataArray[$key]["tot"] = $thisDataArray[$key];
                        $statsDataArray[$key]["min"] = $thisDataArray[$key];
                        $statsDataArray[$key]["max"] = $thisDataArray[$key];
                    }
                }
            }
        }
        if ($i > 0) {
            foreach ($statsDataArray as $key => $value) {
                $statsDataArray[$key]["avg"] = $statsDataArray[$key]["tot"] / $i;

                # compare most recent year to 5 year average
                $thisDataArray = $thisDataArrayRows[0];
                if ($statsDataArray[$key]["avg"] > 0) {
                    $statsDataArray[$key]["diff"] = number_format(100.0 * (($thisDataArray[$key]) - ($statsDataArray[$key]["avg"])) / ($statsDataArray[$key]["avg"]), 1);
                } else {
                    $statsDataArray[$key]["diff"] = 0;
                }
            }
        }
        return $statsDataArray;
    }

    /**
     * output HTML version of the report
     *
     */
    function outputHtmlReport($agg_type, $loc_type, $loc_id, $f_id, $time_period, $uid = null, $sid = null)
    {
        $this->tpl->assign('webroot', '.');
        $mod_agg_type = $agg_type;
        if ($mod_agg_type == null || $mod_agg_type == "") {
            $mod_agg_type = $this->getInferredAggType($time_period);
        }
        # ensure that html version only prints the first f_id in a list
        $f_ids = explode(",", $f_id);
        $f_id_mod = $f_ids[0];
        if ($f_id_mod == "C") { # create cover
            $this->createCoverPage($mod_agg_type, $loc_type, $loc_id, $f_id, $time_period, $uid, $sid);
            $html = $this->tpl->fetch($this->getTemplate($mod_agg_type, $f_id_mod, true));
        } else {
            $this->createReport($mod_agg_type, $loc_type, $loc_id, $f_id_mod, $time_period, $uid, $sid);
            $html = $this->tpl->fetch($this->getTemplate($mod_agg_type, $f_id_mod));
        }
        echo $html;
    }

    /**
     * infer agg_type based on time_period (YYYY or YYYYMM format)
     *
     */
    function getInferredAggType($time_period)
    {
        if (strlen($time_period) == 6) {
            $agg_type = "MONTHLY";
        } else if (strlen($time_period) == 5) {
            $agg_type = "QUARTERLY";
        } else if (strlen($time_period) == 4) {
            $agg_type = "YEARLY";
        } else {
            throw new Exception("Invalid Time Period");
        }
        return $agg_type;
    }

    function getTemplate($agg_type, $f_id, $cover = false)
    {
        if ($f_id == 1000) {
            $template = strtolower("lmi_$agg_type.tpl");
        } else if ($cover) {
            $template = strtolower("cover.tpl");
        } else {
            $template = strtolower("lmi_$agg_type.tpl");
           # $template = strtolower("lmi_" . $agg_type . "_with_summary.tpl");
        }
        return $template;
    }


    /**
     * generate PDF version of the report
     *
     * @param array $formvars the form variables
     */
    function outputPdfReport($agg_type, $loc_type, $loc_id, $f_id, $time_period, $filename, $title, $uid = null, $sid = null)
    {
         $this->tpl->assign('webroot', '..');
        // assign error message
        $mod_agg_type = $agg_type;
        if ($mod_agg_type == null || $mod_agg_type == "") {
            $mod_agg_type = $this->getInferredAggType($time_period);
        }

        $f_ids = explode(",", $f_id);
        $pdf_files = array();
        foreach ($f_ids as $f_id_mod) {
            if ($f_id_mod == "C") { # create cover
                $this->createCoverPage($mod_agg_type, $loc_type, $loc_id, $f_id, $time_period, $uid, $sid);
                $html = $this->tpl->fetch($this->getTemplate($mod_agg_type, $f_id_mod, true));
            } else {
                $this->createReport($mod_agg_type, $loc_type, $loc_id, $f_id_mod, $time_period, $uid, $sid);
                $html = $this->tpl->fetch($this->getTemplate($mod_agg_type, $f_id_mod));
            }
            $save_filename = $filename . "." . $f_id_mod . ".pdf";
            try {
                $wkhtmltopdf = new Wkhtmltopdf(array('path' => RBI_REPORT_DIR . '/public/'));
                $wkhtmltopdf->setTitle($title);
                $wkhtmltopdf->setHtml($html);
                $wkhtmltopdf->setPageSize(Wkhtmltopdf::SIZE_LETTER);
                if (sizeof($f_ids) == 1) {
                    $wkhtmltopdf->output(Wkhtmltopdf::MODE_EMBEDDED, $filename);
                } else {
                    $wkhtmltopdf->output(Wkhtmltopdf::MODE_SAVE, $save_filename);
                }
            } catch (Exception $e) {
                echo $e->getMessage();
            }
            array_push($pdf_files, RBI_REPORT_DIR . 'public/' . $save_filename);
        }
        if (sizeof($f_ids) > 1) {
            try {
                $pdftk = new PDFTK();
                $pdftk->outputCompiledPdf($pdf_files,$filename);
            } catch (Exception $e) {
                echo $e->getMessage();
            }
        }
    }

    /**
     * generate array of time_periods:
     *   $tp_start: time_period to start sequence
     *   $tp_end: time_period to end sequence
     *   $period:  'Y' for year
     *             'M' for month (default)
     */
    function generate_time_periods($tp_start, $tp_end, $period = 'M')
    {
        $retval = array();
        if ($tp_end < $tp_start) {
            return $retval;
        }

        $tp_start_date = $tp_start . "01";
        $tp_gen = $tp_start;
        $i = 0;
        while ($tp_gen < $tp_end) {
            $date_interval = "P" . $i . $period;
            $tp_gen_tms = DateTime::createFromFormat("Ymd", $tp_start_date)->add(new DateInterval($date_interval));
            $tp_gen = $tp_gen_tms->format("Ym");
            $retval[] = $tp_gen;
            $i++;
        }
        return $retval;
    }

    function generate_time_periods_quarterly($tp_start, $tp_end, $period = 'Q')
    {
        $retval = array();
        if ($tp_end < $tp_start) {
            return $retval;
        }

        if ($period == 'Y') {
            $period_add = 10;
        } else {
            $period_add = 1;
        }

        $tp_gen = $tp_start;
        $i = 0;
        while ($tp_gen < $tp_end) {
            if (($period == "Q") && (substr($tp_gen, 4, 1) == "4")) {
                $tp_gen = strval(substr($tp_gen, 0, 4) + 1) . "1";
            } else {
                $tp_gen = $tp_gen + $period_add;
            }
            $retval[] = $tp_gen;
            $i++;
        }
        return $retval;
    }


    /**
     * get generation date
     */
    function getGenerationDate($agg_type, $loc_type, $loc_id, $f_id, $time_period)
    {
        try {
            $table_name = $this->getAggTable($agg_type);
            $query = "select max(last_update_tms) from $table_name where f_id=$f_id and loc_type=$loc_type and loc_id=$loc_id and time_period = $time_period";
            #echo $query . "<BR>\n";

            foreach ($this->pdo->query($query) as $row) {
                $rows[] = $row;
            }
        } catch (PDOException $e) {
            echo "Error!: " . $e->getMessage();
            return false;
        }
        return $rows[0][0];
    }

    function getAggTable($agg_type, $threedim = false)
    {
        $table_name = "facet_aggregations";
        if ($agg_type == "QTD" || $agg_type == "YTD" || $agg_type == "QUARTERLY") {
            $table_name = $table_name . "_" . $agg_type;
        }
        if ($threedim) {
            $table_name = $table_name . "_3fd";
        }
        $table_name = strtolower($table_name);
        return $table_name;
    }

    /**
     * get all data for a single month
     */
    function getPeriodData($agg_type, $loc_type, $loc_id, $f_id, $time_period)
    {
        try {
            $table_name = $this->getAggTable($agg_type);
            $query = "select time_period, num_new_listings, num_pending_listings, num_new_pending_listings, num_sold_listings, sld_avg_sp, sld_avg_sp_per_sqft, sld_med_sp, num_active_listings, sld_avg_dom, sld_med_days_to_contract, sld_med_days_to_settle, sld_avg_sp_to_olp_ratio, if(num_active_listings!=0,round(num_pending_listings/num_active_listings,2),0) contract_ratio from $table_name where f_id=$f_id and loc_type=$loc_type and loc_id=$loc_id and time_period = $time_period";
            #echo $query . "<BR>\n";

            foreach ($this->pdo->query($query) as $row) {
                $rows[] = $row;
            }
        } catch (PDOException $e) {
            echo "Error!: " . $e->getMessage();
            return false;
        }
        return $rows[0];
    }


    function getPeriodFiveYearData($agg_type, $loc_type, $loc_id, $f_id, $time_period)
    {
        try {
            $table_name = $this->getAggTable($agg_type);

            if ($agg_type == "MONTHLY") {
                $time_periods = $this->generate_time_periods(strval($time_period - 400), $time_period, 'Y');
            } else if ($agg_type == "QUARTERLY") {
                $time_periods = $this->generate_time_periods_quarterly(strval($time_period - 50), $time_period, 'Y');
            }
            $time_periods_str = implode(",", $time_periods);

            $query = "select time_period, num_new_listings, num_pending_listings, num_new_pending_listings, num_sold_listings, sld_avg_sp, sld_avg_sp_per_sqft, sld_med_sp, num_active_listings, sld_avg_dom, sld_med_days_to_contract, sld_med_days_to_settle, sld_avg_sp_to_olp_ratio, if(num_active_listings!=0,round(num_pending_listings/num_active_listings,2),0) contract_ratio from $table_name where f_id=$f_id and loc_type=$loc_type and loc_id=$loc_id and time_period in ($time_periods_str) order by time_period desc";
            #echo $query . "<BR>\n";

            $rows = array();
            foreach ($this->pdo->query($query) as $row) {
                $rows[] = $row;
            }
        } catch (PDOException $e) {
            echo "Error!: " . $e->getMessage();
            return false;
        }
        return $rows;
    }

    function getChartData($agg_type, $loc_type, $loc_id, $f_id, $time_period)
    {
        try {
            $table_name = $this->getAggTable($agg_type);
            if ($agg_type == "MONTHLY") {
                $time_periods = $this->generate_time_periods(strval($time_period - 200), $time_period, 'M');
            } else if ($agg_type == "QUARTERLY") {
                $time_periods = $this->generate_time_periods_quarterly(strval($time_period - 20), $time_period, 'Q');
            }
            $time_periods_str = implode(",", $time_periods);

            $query = "select time_period, num_pending_listings, num_new_pending_listings, sld_avg_sp, sld_avg_sp_per_sqft, sld_med_sp, num_sold_listings from $table_name where f_id=$f_id and loc_type=$loc_type and loc_id=$loc_id and time_period in ($time_periods_str) order by time_period";
            #echo $query . "<BR>\n";

            $rows = array();
            foreach ($this->pdo->query($query) as $row) {
                $rows[] = $row;
            }
        } catch (PDOException $e) {
            echo "Error!: " . $e->getMessage();
            return false;
        }
        return $rows;
    }

    function getYTDData($agg_type, $loc_type, $loc_id, $f_id, $time_period)
    {
        try {
            $table_name = $this->getAggTable($agg_type);
            $query = "select time_period, num_new_listings, num_pending_listings, num_new_pending_listings, num_sold_listings, sld_avg_sp, sld_avg_sp_per_sqft, sld_med_sp, num_active_listings, sld_avg_dom, sld_med_days_to_contract, sld_med_days_to_settle, sld_avg_sp_to_olp_ratio from facet_aggregations_ytd where f_id=$f_id and loc_type=$loc_type and loc_id=$loc_id and time_period = $time_period";
            #echo $query . "<BR>\n";

            $rows = array();
            foreach ($this->pdo->query($query) as $row) {
                $rows[] = $row;
            }
        } catch (PDOException $e) {
            echo "Error!: " . $e->getMessage();
            return false;
        }
        return $rows[0];
    }

    function getYTDTwoYearData($agg_type, $loc_type, $loc_id, $f_id, $time_period)
    {
        try {
            $table_name = $this->getAggTable($agg_type);
            $query = "select num_new_listings, num_pending_listings, num_new_pending_listings, num_sold_listings, sld_avg_sp, sld_avg_sp_per_sqft, sld_med_sp, num_active_listings, sld_avg_dom, sld_med_days_to_contract, sld_med_days_to_settle, sld_avg_sp_to_olp_ratio from facet_aggregations_ytd where f_id=$f_id and loc_type=$loc_type and loc_id=$loc_id and time_period in ($time_period, $time_period-100) order by time_period desc";
            #echo $query . "<BR>\n";

            $rows = array();
            foreach ($this->pdo->query($query) as $row) {
                $rows[] = $row;
            }
        } catch (PDOException $e) {
            echo "Error!: " . $e->getMessage();
            return false;
        }
        return $rows;
    }

    function getUserData($uid)
    {
        $url = $this->config->user_data_url . $uid;
        $user_data = file_get_contents($url);
        return json_decode($user_data, true);
    }

    function getUserDataSugar($sid)
    {
        $login = sugar_login();
        $user_data = sugar_get_contact_by_id($sid);
        return $user_data;
    }

    function getLocationData($loc_type, $loc_id)
    {
        try {
            if ($loc_type == 5) {
                $row["l_id"] = substr("00000" . $loc_id, -5, 5);
                $row["lt_id"] = $loc_type;
                $row["l_desc"] = $row["l_id"];
                $row["l_shortname"] = $row["l_id"];
                $rows[0] = $row;
            } else {
                $query = "select l_id, lt_id, case when lt_id=1 and l_shortname is not null then l_shortname else l_desc end as l_desc, lkp_id, l_shortname from location_values where lt_id=$loc_type and l_id=$loc_id";
                foreach ($this->pdo->query($query) as $row) {
                    $rows[] = $row;
                }
                if (count($rows) == 0) {
                    throw new Exception("No Rows Found -- missing data in source.");
                }
            }
        } catch (PDOException $e) {
            echo "Error!: " . $e->getMessage();
            return false;
        }
        return $rows[0];
    }

    function getFacetData()
    {
        try {
            $query = "select * from facet_values where ft_id in (1,2)";
            foreach ($this->pdo->query($query) as $row) {
                $rows[] = $row;
            }
            if (count($rows) == 0) {
                throw new Exception("No Rows Found -- missing data in source.");
            }
        } catch (PDOException $e) {
            echo "Error!: " . $e->getMessage();
            return false;
        }
        return $rows;
    }

    function getLocationTypes()
    {
        try {
            $query = "select * from location_types";
            foreach ($this->pdo->query($query) as $row) {
                $rows[] = $row;
            }
            if (count($rows) == 0) {
                throw new Exception("No Rows Found -- missing data in source.");
            }
        } catch (PDOException $e) {
            echo "Error!: " . $e->getMessage();
            return false;
        }
        return $rows;
    }

    function getLocationValues($lt_id)
    {
        try {
            $query = "select * from location_values where lt_id=$lt_id order by l_id";
            if ($lt_id == 4) {
                $query = "select * from location_values where lt_id=$lt_id order by substr(l_desc,instr(l_desc, ',')+2), l_desc";
            }
            if ($lt_id == 5) {
                $query = "select distinct zip as l_id, zip as l_desc from ZIP_MASTER_LIST order by zip";
            }
            foreach ($this->pdo->query($query) as $row) {
                $rows[] = $row;
            }
            if (count($rows) == 0) {
                throw new Exception("No Rows Found -- missing data in source.");
            }
        } catch (PDOException $e) {
            echo "Error!: " . $e->getMessage();
            return false;
        }
        return $rows;
    }

    function getReportMonths()
    {
        try {
            $query = "select distinct time_period from facet_aggregations where time_period>2500  order by time_period desc";
            foreach ($this->pdo->query($query) as $row) {
                $rows[] = $row;
            }
            if (count($rows) == 0) {
                throw new Exception("No Rows Found -- missing data in source.");
            }
        } catch (PDOException $e) {
            echo "Error!: " . $e->getMessage();
            return false;
        }
        return $rows;
    }

    function getReportQuarters()
    {
        try {
            $query = "select distinct time_period from facet_aggregations_quarterly order by time_period desc";
            foreach ($this->pdo->query($query) as $row) {
                $rows[] = $row;
            }
            if (count($rows) == 0) {
                throw new Exception("No Rows Found -- missing data in source.");
            }
        } catch (PDOException $e) {
            print "Error!: " . $e->getMessage();
            return false;
        }
        return $rows;
    }
}
